# apps/services/models.py

from decimal import Decimal
from django.db import models
from django.utils.translation import gettext_lazy as _
from apps.agreement.models import Agreement


class Services(models.Model):
    FLOOR_CHOICES = (
        (1, "First floor"),
        (2, "Second Floor"),
        (3, "Third Floor"),
        (4, "Fourth Floor"),
        (5, "Fifth Floor"),
        (6, "UnderGround"),
    )

    MONTH_CHOICES = (
        (1, "حمل"), (2, "ثور"), (3, "جوزا"), (4, "سرطان"),
        (5, "اسد"), (6, "سنبله"), (7, "میزان"), (8, "عقرب"),
        (9, "قوس"), (10, "جدی"), (11, "دلو"), (12, "حوت"),
    )

    floor = models.CharField(_("Floor"), choices=FLOOR_CHOICES, max_length=50)
    time = models.CharField(_("Time"), choices=MONTH_CHOICES, max_length=250)
    year = models.CharField(max_length=255)
    total = models.DecimalField(
        _("Total Service Fee"), max_digits=10, decimal_places=2, default=Decimal("0.00")
    )
    is_approved = models.BooleanField(default=False)
    customers_list = models.JSONField(default=dict)
    created_at = models.DateTimeField(auto_now_add=True, editable=False)
    updated_at = models.DateTimeField(auto_now=True, editable=False)

    class Meta:
        verbose_name = _("Service")
        verbose_name_plural = _("Services")
        ordering = ['-year', '-time', 'floor']

    def __str__(self):
        return f"Services - Floor {self.get_floor_display()} - {self.get_time_display()} {self.year}"

    # --- UPDATED METHOD WITH CORRECT LOGIC ---
    def get_initial_customers_data(self):
        if not self.floor:
            return {}, Decimal("0.00")

        # --- NEW: Find the previous month's service record first ---
        try:
            current_month = int(self.time)
            current_year = int(self.year)
            if current_month == 1:
                previous_month_time = 12
                previous_month_year = current_year - 1
            else:
                previous_month_time = current_month - 1
                previous_month_year = current_year
            
            # Query for the previous service record for this specific floor
            previous_service_record = Services.objects.filter(
                floor=self.floor,
                time=str(previous_month_time),
                year=str(previous_month_year)
            ).first()
        except (ValueError, TypeError):
            # If year/time are invalid, there's no previous record
            previous_service_record = None
        # --- END NEW LOGIC ---
        
        try:
            agreements = Agreement.objects.filter(status="Active", floor=self.floor)
        except Exception as e:
            print(f"Error fetching agreements for floor {self.floor} (Services): {e}")
            return {}, Decimal("0.00")

        customers_data = {}
        total_service = Decimal("0.00")

        for agreement in agreements:
            if not agreement.customer:
                continue

            customer_service = getattr(agreement, 'service', Decimal("0.00"))
            if customer_service is None:
                customer_service = Decimal("0.00")
            else:
                 customer_service = Decimal(str(customer_service))

            # --- NEW: Determine the correct previous meter number ---
            final_previous_reading = ""
            customer_id_str = str(agreement.customer.id)

            # 1. Prioritize the previous service record if it exists
            if previous_service_record and isinstance(previous_service_record.customers_list, dict):
                previous_customer_data = previous_service_record.customers_list.get(customer_id_str)
                if previous_customer_data:
                    # Use the 'current_meter_number' from the *last* month's bill
                    reading = previous_customer_data.get('current_meter_number')
                    if reading and str(reading).strip(): # Check if it's not None or an empty string
                        final_previous_reading = reading
            
            # 2. If no reading was found, fallback to the main Agreement
            if not final_previous_reading:
                final_previous_reading = agreement.meter_number or ""
            # --- END NEW LOGIC ---

            taken = Decimal("0.00")
            meter_bill = Decimal("0.00")
            total_bill = customer_service + meter_bill
            remainder = total_bill - taken

            customer_data = {
                "service": float(customer_service),
                "taken": float(taken),
                "is_approved": self.is_approved,
                "shop": str(agreement.shop) if hasattr(agreement, 'shop') and agreement.shop else None,
                "previous_meter_number": final_previous_reading, # Use the correctly determined value
                "current_meter_number": "",
                "meter_bill": float(meter_bill),
                "total_bill": float(total_bill),
                "remainder": float(remainder),
                "description": "",
            }
            
            customers_data[customer_id_str] = customer_data
            total_service += customer_service

        return customers_data, total_service

    def save(self, *args, **kwargs):
        is_new = self.pk is None
        if is_new:
            customers_data, total_service = self.get_initial_customers_data()
            self.customers_list = customers_data
            self.total = total_service
            for cust_id in self.customers_list:
                if isinstance(self.customers_list[cust_id], dict):
                     self.customers_list[cust_id]['is_approved'] = self.is_approved
        else:
            if isinstance(self.customers_list, dict):
                current_total_service = sum(
                    Decimal(str(cust_data.get('service', 0)))
                    for cust_data in self.customers_list.values()
                    if isinstance(cust_data, dict)
                )
                self.total = current_total_service
            else:
                 self.customers_list = {}
                 self.total = Decimal("0.00")

        super().save(*args, **kwargs)