# apps/services/serializers.py

from decimal import Decimal
from rest_framework import serializers
from .models import Services
from apps.agreement.models import Agreement # <-- ADDED: Import the Agreement model

class ServiceSerializer(serializers.ModelSerializer):
    customers_list = serializers.JSONField(read_only=False, required=False)
    
    total_service_fee = serializers.DecimalField(max_digits=10, decimal_places=2, read_only=True, source='total')
    grand_total_bill = serializers.SerializerMethodField()
    total_taken = serializers.SerializerMethodField()
    total_remainder = serializers.SerializerMethodField()

    class Meta:
        model = Services
        fields = (
            "id",
            "floor",
            "time",
            "year",
            "is_approved",
            "customers_list",
            "total_service_fee",
            "grand_total_bill",
            "total_taken",
            "total_remainder",
            "created_at",
            "updated_at",
        )
        read_only_fields = ('created_at', 'updated_at')

    def _parse_decimal(self, value, default=Decimal("0.00")):
        try:
            return Decimal(str(value)) if value is not None else default
        except (ValueError, TypeError):
            return default

    def get_grand_total_bill(self, obj):
        if isinstance(obj.customers_list, dict):
            return sum(
                self._parse_decimal(customer.get("total_bill", 0))
                for customer in obj.customers_list.values() if isinstance(customer, dict)
            )
        return Decimal("0.00")

    def get_total_taken(self, obj):
        if isinstance(obj.customers_list, dict):
            return sum(
                self._parse_decimal(customer.get("taken", 0))
                for customer in obj.customers_list.values() if isinstance(customer, dict)
            )
        return Decimal("0.00")

    def get_total_remainder(self, obj):
        if isinstance(obj.customers_list, dict):
            return sum(
                self._parse_decimal(customer.get("remainder", 0))
                for customer in obj.customers_list.values() if isinstance(customer, dict)
            )
        return Decimal("0.00")

    def update(self, instance, validated_data):
        instance.floor = validated_data.get("floor", instance.floor)
        instance.time = validated_data.get("time", instance.time)
        instance.year = validated_data.get("year", instance.year)
        instance.is_approved = validated_data.get("is_approved", instance.is_approved)

        customers_data_from_request = validated_data.get("customers_list")

        if customers_data_from_request is not None:
            if not isinstance(instance.customers_list, dict):
                instance.customers_list = {}

            for customer_id, customer_update_data in customers_data_from_request.items():
                if not isinstance(customer_update_data, dict):
                    continue

                customer_id_str = str(customer_id)
                existing_customer_data = instance.customers_list.get(customer_id_str, {})
                
                new_service = self._parse_decimal(customer_update_data.get('service', existing_customer_data.get('service', 0)))
                new_meter_bill = self._parse_decimal(customer_update_data.get('meter_bill', existing_customer_data.get('meter_bill', 0)))
                new_taken = self._parse_decimal(customer_update_data.get('taken', existing_customer_data.get('taken', 0)))
                new_current_meter = customer_update_data.get('current_meter_number', existing_customer_data.get('current_meter_number', ''))
                new_description = customer_update_data.get('description', existing_customer_data.get('description', ''))
                new_is_approved = customer_update_data.get('is_approved', instance.is_approved)
                if 'current_meter_number' in customer_update_data:
                    try:
                        # Find the customer's active agreement
                        agreement_to_update = Agreement.objects.filter(
                            customer__id=int(customer_id),
                            status="Active"
                        ).first()

                        # If an active agreement is found, update its meter number
                        if agreement_to_update:
                            agreement_to_update.meter_number = new_current_meter
                            agreement_to_update.save()
                    except (ValueError, TypeError):
                        print(f"Warning: Could not update agreement for invalid customer_id '{customer_id}'")
                    except Exception as e:
                        print(f"Error updating agreement for customer {customer_id}: {e}")


                new_total_bill = new_service + new_meter_bill
                new_remainder = new_total_bill - new_taken
                updated_data_for_customer = {
                    **existing_customer_data,
                    "service": float(new_service),
                    "meter_bill": float(new_meter_bill),
                    "total_bill": float(new_total_bill),
                    "taken": float(new_taken),
                    "remainder": float(new_remainder),
                    "current_meter_number": new_current_meter,
                    "description": new_description,
                    "is_approved": new_is_approved,
                }

                instance.customers_list[customer_id_str] = updated_data_for_customer

        instance.save()
        return instance