# apps/units/serializers.py
import json
from decimal import Decimal, InvalidOperation
from rest_framework import serializers
from django.utils.translation import gettext_lazy as _
from .models import Unit, UnitBill # Unit model is already imported

class UnitSerializer(serializers.ModelSerializer):
    # ... (no changes needed here) ...
    """Serializer for the Unit model."""
    class Meta:
        model = Unit
        fields = [
            "id", "customer_name", "customer_father_name", "unit_number",
            "services_description", "service_charge", "current_water_reading",
            "current_electricity_reading", "status", "created_at", "updated_at",
        ]
        read_only_fields = ('id', 'created_at', 'updated_at')

    def validate(self, data):
        status = data.get('status', self.instance.status if self.instance else None)
        customer_name = data.get('customer_name', self.instance.customer_name if self.instance else None)
        if status == Unit.Status.OCCUPIED and not customer_name:
            raise serializers.ValidationError({"customer_name": _("Occupier Name is required when status is Occupied.")})
        unit_number = data.get('unit_number')
        if unit_number: # Check uniqueness on create/update if needed
           query = Unit.objects.filter(unit_number=unit_number)
           if self.instance:
               query = query.exclude(pk=self.instance.pk)
           if query.exists():
               raise serializers.ValidationError({"unit_number": _("Unit with this number already exists.")})
        return data

    def to_representation(self, instance):
        data = super().to_representation(instance)
        data["service_charge"] = float(instance.service_charge or 0.0)
        data["current_water_reading"] = float(instance.current_water_reading or 0.0)
        data["current_electricity_reading"] = float(instance.current_electricity_reading or 0.0)
        return data


class UnitBillSerializer(serializers.ModelSerializer):
    total_paid = serializers.FloatField(read_only=True, default=0.0)
    total_remainder = serializers.FloatField(read_only=True, default=0.0)
    unit_list = serializers.SerializerMethodField(read_only=True) # Formatted list
    unit_details_list = serializers.JSONField(read_only=True) # Raw data output for modal

    class Meta:
        model = UnitBill
        fields = (
            "id", "month", "year", "total",
            "unit_details_list", # Included for reading by modal
            "unit_list",         # Optional formatted list
            "created_at", "updated_at",
            "total_paid", "total_remainder",
        )
        read_only_fields = (
            'id', 'created_at', 'updated_at', 'total',
            'unit_list', 'unit_details_list',
            'total_paid', 'total_remainder'
        )

    def _parse_decimal(self, value, default=Decimal("0.00")):
        # Make parsing slightly more robust
        try:
            if value is None or str(value).strip() == '':
                return default
            # Handle potential float input from frontend JSON parsing
            return Decimal(str(value))
        except (InvalidOperation, ValueError, TypeError):
            return default

    def get_unit_list(self, instance):
        # ... (no changes needed here) ...
        output_list = {}
        details = instance.unit_details_list
        if not isinstance(details, dict): return {}
        for unit_id_str, data in details.items():
            if not isinstance(data, dict): continue
            charge = self._parse_decimal(data.get("service_charge"))
            paid = self._parse_decimal(data.get("amount_paid"))
            remainder = charge - paid
            output_data = {
                "unit_id": data.get("unit_id"), "customer_name": data.get("customer_name"),
                "customer_father_name": data.get("customer_father_name"), "unit_number": data.get("unit_number"),
                "services": int(charge), "service_description_text": data.get("services_description"),
                "current_waterMeter": float(self._parse_decimal(data.get("current_water_reading"))),
                "current_electricityMeter": float(self._parse_decimal(data.get("current_electricity_reading"))),
                "taken": float(paid), "remainder": float(remainder), "description": data.get("description", ""),
            }
            output_list[unit_id_str] = output_data
        return output_list

    def update(self, instance, validated_data):
        instance.month = validated_data.get("month", instance.month)
        instance.year = validated_data.get("year", instance.year)

        request_data = self.context['request'].data
        incoming_unit_data = request_data.get("unit_details_list", None)

        current_unit_details = instance.unit_details_list
        if not isinstance(current_unit_details, dict):
            try: current_unit_details = json.loads(current_unit_details) if isinstance(current_unit_details, str) else {}
            except json.JSONDecodeError: current_unit_details = {}
        instance.unit_details_list = current_unit_details # Work directly on instance data

        # --- START: Logic to update Unit model ---
        units_to_update = {} # Store Unit instances that need saving
        unit_update_fields = {} # Track which fields changed for each unit
        # --- END: Logic to update Unit model ---

        if isinstance(incoming_unit_data, dict):
            details_list_changed = False
            for unit_id_str, unit_data_in in incoming_unit_data.items():
                 unit_id_str = str(unit_id_str) # Ensure key is string
                 if not isinstance(unit_data_in, dict): continue

                 existing_unit_data = current_unit_details.get(unit_id_str)
                 if existing_unit_data is None:
                     print(f"Warning: Unit ID {unit_id_str} from request not found in existing bill details.")
                     continue # Skip if this unit wasn't originally in the bill snapshot

                 # --- Get the actual Unit ID (important!) ---
                 # Prefer the stored unit_id if available, fallback to key
                 unit_pk = existing_unit_data.get("unit_id")
                 if unit_pk is None:
                     try:
                         unit_pk = int(unit_id_str)
                     except ValueError:
                         print(f"Warning: Could not determine valid unit PK from key '{unit_id_str}'. Skipping meter update for this entry.")
                         continue
                 # ---

                 unit_updated = False
                 charge_or_paid_updated = False
                 meter_reading_updated = False # Track meter changes specifically

                 # Update Service Charge (existing logic)
                 new_charge_str = unit_data_in.get("service_charge")
                 if new_charge_str is not None:
                     new_charge = self._parse_decimal(new_charge_str)
                     if new_charge != self._parse_decimal(existing_unit_data.get("service_charge")):
                         existing_unit_data["service_charge"] = str(new_charge) # Store as string in JSON
                         unit_updated = True; charge_or_paid_updated = True

                 # Update Amount Paid (existing logic)
                 new_paid_str = unit_data_in.get("amount_paid")
                 if new_paid_str is not None:
                     new_paid = self._parse_decimal(new_paid_str)
                     if new_paid != self._parse_decimal(existing_unit_data.get("amount_paid")):
                         existing_unit_data["amount_paid"] = str(new_paid) # Store as string in JSON
                         unit_updated = True; charge_or_paid_updated = True

                 # Update Description (existing logic)
                 new_desc = unit_data_in.get("description")
                 if new_desc is not None and str(new_desc) != existing_unit_data.get("description", ""):
                     existing_unit_data["description"] = str(new_desc)
                     unit_updated = True

                 # --- START: Update Meter Readings & Mark Unit for Update ---
                 new_water_str = unit_data_in.get("current_water_reading")
                 if new_water_str is not None:
                     new_water = self._parse_decimal(new_water_str)
                     if new_water != self._parse_decimal(existing_unit_data.get("current_water_reading")):
                         existing_unit_data["current_water_reading"] = str(new_water) # Update JSON snapshot
                         unit_updated = True
                         meter_reading_updated = True
                         # Prepare Unit instance for update
                         if unit_pk not in units_to_update:
                             try:
                                 units_to_update[unit_pk] = Unit.objects.get(pk=unit_pk)
                                 unit_update_fields[unit_pk] = set()
                             except Unit.DoesNotExist:
                                 print(f"Warning: Unit with PK {unit_pk} not found. Cannot update meter readings.")
                                 units_to_update[unit_pk] = None # Mark as not found
                         if units_to_update.get(unit_pk):
                             units_to_update[unit_pk].current_water_reading = new_water
                             unit_update_fields[unit_pk].add('current_water_reading')


                 new_elec_str = unit_data_in.get("current_electricity_reading")
                 if new_elec_str is not None:
                     new_elec = self._parse_decimal(new_elec_str)
                     if new_elec != self._parse_decimal(existing_unit_data.get("current_electricity_reading")):
                         existing_unit_data["current_electricity_reading"] = str(new_elec) # Update JSON snapshot
                         unit_updated = True
                         meter_reading_updated = True
                         # Prepare Unit instance for update
                         if unit_pk not in units_to_update:
                             try:
                                 units_to_update[unit_pk] = Unit.objects.get(pk=unit_pk)
                                 unit_update_fields[unit_pk] = set()
                             except Unit.DoesNotExist:
                                 print(f"Warning: Unit with PK {unit_pk} not found. Cannot update meter readings.")
                                 units_to_update[unit_pk] = None # Mark as not found
                         if units_to_update.get(unit_pk):
                            units_to_update[unit_pk].current_electricity_reading = new_elec
                            unit_update_fields[unit_pk].add('current_electricity_reading')
                 # --- END: Update Meter Readings & Mark Unit for Update ---

                 if unit_updated:
                     details_list_changed = True
                     if charge_or_paid_updated: # Update remainder if charge/paid changed
                         current_charge = self._parse_decimal(existing_unit_data.get("service_charge"))
                         current_paid = self._parse_decimal(existing_unit_data.get("amount_paid"))
                         existing_unit_data["remainder"] = str(current_charge - current_paid)

            if details_list_changed:
                # Recalculate total based on potentially changed service charges
                instance.recalculate_total_from_details()

        # --- START: Save Updated Unit Instances ---
        for unit_pk, unit_instance in units_to_update.items():
            if unit_instance and unit_pk in unit_update_fields:
                fields_to_save = list(unit_update_fields[unit_pk])
                if fields_to_save:
                    print(f"Saving Unit {unit_pk} with fields: {fields_to_save}")
                    unit_instance.save(update_fields=fields_to_save)
        # --- END: Save Updated Unit Instances ---

        # Save the UnitBill instance itself (with updated details_list and total)
        instance.save()
        return instance

    def to_representation(self, instance):
        # ... (no changes needed here) ...
        data = super().to_representation(instance) # Includes unit_details_list now
        total_charge = Decimal("0.00")
        total_paid_calc = Decimal("0.00")
        unit_details_repr = data.get("unit_details_list", {})
        if not isinstance(unit_details_repr, dict): unit_details_repr = {}
        for unit_id, unit_data in unit_details_repr.items():
            if isinstance(unit_data, dict):
                charge = self._parse_decimal(unit_data.get("service_charge"))
                paid = self._parse_decimal(unit_data.get("amount_paid"))
                total_charge += charge
                total_paid_calc += paid
        total_remainder_calc = total_charge - total_paid_calc

        data["total_paid"] = float(total_paid_calc)
        data["total_remainder"] = float(total_remainder_calc)
        data["total"] = float(instance.total or 0.0)
        data["unit_list"] = self.get_unit_list(instance) # Add formatted list

        return data